# Copyright 2020 Red Hat, Inc.
# All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

from unittest import mock
import urllib

from tenacity import RetryError
from tenacity import wait_fixed

from config_tempest import constants as C
from config_tempest.services.octavia import LoadBalancerService
from config_tempest.tests.base import BaseServiceTest


class TestOctaviaService(BaseServiceTest):
    def setUp(self):
        super(TestOctaviaService, self).setUp()
        self.conf = self._get_conf("v2", "v3")
        self.clients = self._get_clients(self.conf)
        self.Service = LoadBalancerService("ServiceName",
                                           "ServiceType",
                                           self.FAKE_URL + "v2.0/",
                                           self.FAKE_TOKEN,
                                           disable_ssl_validation=False)
        self.Service.client = self.FakeServiceClient(
            services={"services": [{"name": "octavia", "enabled": True}]}
        )
        self.conf.set("identity", "region", "regionOne")
        self._fake_service_do_get_method(self.FAKE_LBAAS_PROVIDERS)

    def test_list_drivers(self):
        expected_resp = [
            "amphora:The Octavia Amphora driver.",
            "octavia:Deprecated alias of the Octavia driver.",
        ]
        providers = self.Service.list_drivers()
        self.assertEqual(len(providers), len(expected_resp))

    @mock.patch("config_tempest.services.services.Services.is_service")
    def test_octavia_service_post_configuration(self, mock_is_service):
        mock_is_service.return_value = True
        self.Service.post_configuration(self.conf, mock_is_service)
        self.assertEqual(self.conf.get("load_balancer", "member_role"),
                         "member")
        self.assertEqual(self.conf.get("load_balancer", "region"),
                         "regionOne")
        self.assertEqual(self.conf.get("load_balancer",
                                       "enabled_provider_drivers"),
                         ("amphora:The Octavia Amphora driver.,"
                          "octavia:Deprecated alias of the Octavia driver."),
                         )

    @mock.patch("urllib.request.urlopen")
    @mock.patch("os.path.exists")
    def test_octavia__download_file(self,
                                    mock_path_exists,
                                    mock_urlopen):
        mock_url = mock.Mock()
        mock_destination = mock.Mock()
        mock_data = mock.Mock(name='Fake data')

        # File already exists
        mock_path_exists.return_value = True
        self.Service._download_file(mock_url, mock_destination)
        mock_urlopen.assert_not_called()

        # File doesn't exist, normal path
        mock_path_exists.return_value = False

        mock_response = mock.MagicMock()
        mock_response.read.return_value = mock_data
        mock_urlopen.return_value = mock_response

        mock_open = mock.mock_open()
        with mock.patch("builtins.open", mock_open):
            self.Service._download_file(mock_url, mock_destination)
        mock_urlopen.assert_called_once_with(mock_url)
        mock_open.assert_called_once_with(mock_destination, "wb")
        handle = mock_open()
        handle.write.assert_called_once_with(mock_data)

        mock_urlopen.reset_mock()

        # File doesn't exist, with 2 URLErrors then it passes
        mock_path_exists.return_value = False

        mock_response = mock.MagicMock()
        mock_response.read.return_value = mock_data
        mock_urlopen.side_effect = [
            urllib.error.URLError(reason="reason1"),
            urllib.error.URLError(reason="reason2"),
            mock_response]

        mock_open = mock.mock_open()
        with mock.patch("builtins.open", mock_open):
            # override tenacity.retry wait param
            with mock.patch.object(self.Service._download_file.retry,
                                   "wait", wait_fixed(0)):
                self.Service._download_file(mock_url, mock_destination)
        mock_urlopen.assert_called_with(mock_url)
        mock_open.assert_called_once_with(mock_destination, "wb")
        handle = mock_open()
        handle.write.assert_called_once_with(mock_data)

        # File doesn't exist, with URLErrors
        mock_path_exists.return_value = False

        mock_urlopen.side_effect = [
            urllib.error.URLError(reason="reason1"),
            urllib.error.URLError(reason="reason2"),
            urllib.error.URLError(reason="reason3"),
            urllib.error.URLError(reason="reason4")]

        mock_open = mock.mock_open()
        with mock.patch("builtins.open", mock_open):
            # override tenacity.retry wait param
            with mock.patch.object(self.Service._download_file.retry,
                                   "wait", wait_fixed(0)):
                self.assertRaises(RetryError,
                                  self.Service._download_file,
                                  mock_url,
                                  mock_destination)
        mock_urlopen.assert_called_with(mock_url)
        mock_open.assert_not_called()

    @mock.patch("config_tempest.services.octavia.LoadBalancerService."
                "_download_file")
    @mock.patch("os.chmod")
    def test_octavia_get_test_server_application(self,
                                                 mock_chmod,
                                                 mock_download_file):
        # test_server_remote_url not configured
        self.Service.get_test_server_application(self.conf)
        mock_download_file.assert_not_called()

        # test_server_remote_url set
        location = "dummy://location"
        self.conf.set("load_balancer", "test_server_remote_url",
                      location)
        self.Service.get_test_server_application(self.conf)
        mock_download_file.assert_called_once_with(
            location, C.DEFAULT_OCTAVIA_TEST_SERVER_FILE)
        mock_chmod.assert_called_once_with(
            C.DEFAULT_OCTAVIA_TEST_SERVER_FILE, 0o755)

        self.assertEqual(self.conf.get("load_balancer", "test_server_path"),
                         C.DEFAULT_OCTAVIA_TEST_SERVER_FILE)
